//
//  The template was designed by Hưng Nguyễn. 
//  Website, application (native/flutter), hosting, SEO.
//  Email: hungnguyen.it36@gmail.com
//

import Foundation
#if canImport(FoundationNetworking)
import FoundationNetworking
#endif

import RxSwift

enum HttpMethod:String{
    case post = "POST"
    case get = "GET"
    case put = "PUT"
    case del = "DELETE"
}

public enum ContentType: String {
    case urlFormEncoded = "application/x-www-form-urlencoded"
    case applicationJson = "application/json"
    case applicationRaw = "application/x-www-form-urlencoded; charset=utf-8"
}

typealias ResponseBlock = (_ error:Error?,_ response:[String: Any]?) ->()
typealias ErrorBlock = (_ error:NSError?) ->()
typealias UploadProgessBlock = ((_ progess:Double)->Void)

//MARK: Base
class APIRequest:NSObject {
    
    var enableSSL:Bool = false //_USING_HTTPS_PROTOCOL_
    var boolEncoding = BoolEncoding.numeric
    var arrayEncoding = ArrayEncoding.brackets
    let ctrl = "\r\n"
    lazy var globalQueue:OperationQueue = {
        let queue = OperationQueue()
        queue.name = "com.khohatsi.APIRequest.OperationQueue"
        return queue
    }()
    
    func requestRx(url:String, method:HttpMethod, contentType: ContentType = .urlFormEncoded, allHTTPHeaderFields:[String:String] = [:], params:[String:Any] = [:]) -> Observable<[String: Any]>{
        return Observable.create { (observer) -> Disposable in
            self.request(url: url, method: method, contenType: contentType, allHTTPHeaderFields: allHTTPHeaderFields, params: params, { (error, json) in
                if let error = error {
                    observer.onError(error)
                } else if let json = json {
                    observer.onNext(json)
                    observer.onCompleted()
                } else {
                    observer.onError(TrackerError.selfError)
                }
            }) { (error) in
                if let error = error {
                    observer.onError(error)
                }
            }
            return Disposables.create()
        }
    }
}

//MARK: Rest Request
extension APIRequest {
    func request(url:String, method:HttpMethod, contenType: ContentType = .urlFormEncoded, allHTTPHeaderFields:[String:String] = ["Content-Type":"application/x-www-form-urlencoded"], params:[String:Any] = [:], _ completion:@escaping ResponseBlock){
        switch  method {
        case .post:
            post(url: url, method: .post, contenType: contenType, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, nil)
        case .get:
            get(url: url, method: .get, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, nil)
        case .put:
            post(url: url, method: .put, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, nil)
        case .del:
            delete(url: url, method: .del, contenType: contenType, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, nil)
        }
    }
    
    func request(url:String, method:HttpMethod, contenType: ContentType = .urlFormEncoded, allHTTPHeaderFields:[String:String] = ["Content-Type":"application/x-www-form-urlencoded"], params:[String:Any] = [:], jsonData: Data? = nil, _ completion:@escaping ResponseBlock, errorHandler:ErrorBlock?){
        switch  method {
        case .post:
            post(url: url, method: .post, contenType: contenType, allHTTPHeaderFields: allHTTPHeaderFields, params: params, dataJson: jsonData , completion, errorHandler)
        case .put:
            post(url: url, method: .put, contenType: contenType, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, errorHandler)
        case .get:
            get(url: url, method: .get, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, errorHandler)
        case .del:
            post(url: url, method: .del, contenType: contenType, allHTTPHeaderFields: allHTTPHeaderFields, params: params, completion, errorHandler)
        }
    }
    
    func get(url:String, method:HttpMethod,allHTTPHeaderFields:[String:String], params: [String: Any],_ completion:@escaping ResponseBlock, _ errorHandlerBlock:ErrorBlock?) {
        let urlComponents = NSURLComponents(string: url)
        var items = [URLQueryItem]()
        for (key,value) in params {
               items.append(URLQueryItem(name: key, value: ("\(value)")))
        }
        urlComponents?.queryItems = items
        
        guard let urlRequest =  urlComponents?.url else {
            let error = NSError(domain:"1000", code:404, userInfo:nil)
            completion(error,nil)
            return
        }

        var request = URLRequest(url: urlRequest)
        request.httpMethod = method.rawValue
        request.allHTTPHeaderFields = allHTTPHeaderFields
//        request.addValue(X_TENANT, forHTTPHeaderField: "X-TENANT")
//        request.addValue(UserDataManager.shared().deviceID, forHTTPHeaderField: "deviceId")
//        request.addValue(UserDataManager.shared().token, forHTTPHeaderField: "token")
//        request.addValue(SupportFunction.getAppVersionString(), forHTTPHeaderField: "appVersion")
//        request.addValue(APP_PLATFORM, forHTTPHeaderField: "app-platform")
    
        let config = URLSessionConfiguration.default
        
        let session = URLSession(configuration: config, delegate: self, delegateQueue:globalQueue)
        
        print("curl command: " + request.cURL)
        
        session.dataTask(with: request) { (data, response, error) in
            DispatchQueue.main.async {
                if let data = data {
                    do {
                        let dict = try JSONSerialization.jsonObject(with: data, options: [.allowFragments])
                        completion(nil, dict)
                    } catch let error {
                        APIRequestHandleError.shared().handleError(error)
                        completion(nil, nil)
                    }
                }else{
                    APIRequestHandleError.shared().handleError(error)
                    completion(nil, nil)
                }
            }
        }.resume()
    }
    
    func post(url:String, method:HttpMethod, contenType: ContentType = .urlFormEncoded, allHTTPHeaderFields:[String:String], params:[String:Any] = [:], dataJson: Data? = nil, _ completion:@escaping ResponseBlock,_ errorHandlerBlock:ErrorBlock?){
        guard let urlRequest =  URL(string: url) else {
            let error = NSError(domain:"1000", code:404, userInfo:nil)
            completion(error,nil)
            return
            
        }
        
        var data: Data?
        if contenType == .applicationJson {
            data = try?  JSONSerialization.data(withJSONObject: params, options: [])
        } else {
            data = query(params).data(using: .utf8, allowLossyConversion: false)
        }
        
        var allHeader = allHTTPHeaderFields
        allHeader["Content-Type"] = contenType.rawValue
        var request = URLRequest(url: urlRequest)
        request.httpMethod = method.rawValue
        request.httpBody = dataJson != nil ? dataJson! : data
        request.allHTTPHeaderFields = allHeader
//        request.addValue(X_TENANT, forHTTPHeaderField: "X-TENANT")
//        request.addValue(UserDataManager.shared().deviceID, forHTTPHeaderField: "deviceId")
//        request.addValue(UserDataManager.shared().token, forHTTPHeaderField: "token")
//        request.addValue(SupportFunction.getAppVersionString(), forHTTPHeaderField: "appVersion")
//        request.addValue(APP_PLATFORM, forHTTPHeaderField: "app-platform")
        request.timeoutInterval = 300
        
        let config = URLSessionConfiguration.default
        let session = URLSession(configuration: config, delegate: self, delegateQueue:globalQueue)
        
        print("curl command: " + (request.cURL.count > 5000 ? "" : request.cURL))
        
        session.dataTask(with: request) { (data, response, error) in
            DispatchQueue.main.async {
                if let data = data {
                    do {
                        let dict = try JSONSerialization.jsonObject(with: data, options: [.allowFragments])
                        
                        completion(nil, json)
                        
                    }catch let error {
                        APIRequestHandleError.shared().handleError(error)
                        completion(nil, nil)
                    }
                }else{
                    APIRequestHandleError.shared().handleError(error)
                    completion(nil, nil)
                }
            }
        }.resume()
    }
    
    func delete(url:String, method:HttpMethod, contenType: ContentType = .urlFormEncoded, allHTTPHeaderFields:[String:String], params:[String:Any] = [:], _ completion:@escaping ResponseBlock,_ errorHandlerBlock:ErrorBlock?){
        guard let urlRequest =  URL(string: url) else {
            let error = NSError(domain:"1000", code:404, userInfo:nil)
            completion(error,nil)
            return
            
        }
        
        var data: Data?
        if contenType == .applicationJson {
            data = try?  JSONSerialization.data(withJSONObject: params, options: .prettyPrinted)
        } else {
            data = query(params).data(using: .utf8, allowLossyConversion: false)
        }
        
        var allHeader = allHTTPHeaderFields
        allHeader["Content-Type"] = contenType.rawValue
        var request = URLRequest(url: urlRequest)
        request.httpMethod = method.rawValue
        request.httpBody = data
        request.allHTTPHeaderFields = allHeader
//        request.addValue(X_TENANT, forHTTPHeaderField: "X-TENANT")
//        request.addValue(UserDataManager.shared().deviceID, forHTTPHeaderField: "deviceId")
//        request.addValue(UserDataManager.shared().token, forHTTPHeaderField: "token")
//        request.addValue(SupportFunction.getAppVersionString(), forHTTPHeaderField: "appVersion")
//        request.addValue(APP_PLATFORM, forHTTPHeaderField: "app-platform")
        request.timeoutInterval = 300
        
        let config = URLSessionConfiguration.default
        let session = URLSession(configuration: config, delegate: self, delegateQueue:globalQueue)
        
        print("curl command: " + (request.cURL.count > 5000 ? "" : request.cURL))
        
        session.dataTask(with: request) { (data, response, error) in
            DispatchQueue.main.async {
                if let data = data {
                    do {
                        let dict = try JSONSerialization.jsonObject(with: data, options: [])
                        completion(nil, dict)
                    }catch let error {
                        APIRequestHandleError.shared().handleError(error)
                        completion(nil, nil)
                    }
                }else{
                    APIRequestHandleError.shared().handleError(error)
                    completion(nil, nil)
                }
            }
        }.resume()
    }

}

//MARK: Multipart Request
extension APIRequest {
    func requestUploadMultipart(url:String,method:HttpMethod, multipartDatas:[MultipartData] = [], uploadProgessBlock:@escaping UploadProgessBlock = {(_) in}, _ completion:@escaping ResponseBlock,_ errorHandlerBlock:ErrorBlock?) {
        let url = URL(string: url)
        let config = URLSessionConfiguration.default
        let session = URLSession(configuration: config, delegate: URLSessionDataProxyDelegate(uploadProgessBlock: uploadProgessBlock), delegateQueue:globalQueue)
        
        // Set the URLRequest to POST and to the specified URL
        var urlRequest = URLRequest(url: url!)
        urlRequest.httpMethod = method.rawValue
//        urlRequest.addValue(X_TENANT, forHTTPHeaderField: "X-TENANT")
//        urlRequest.addValue(UserDataManager.shared().deviceID, forHTTPHeaderField: "deviceId")
//        urlRequest.addValue(UserDataManager.shared().token, forHTTPHeaderField: "token")
//        urlRequest.addValue(SupportFunction.getAppVersionString(), forHTTPHeaderField: "appVersion")
//        urlRequest.addValue(APP_PLATFORM, forHTTPHeaderField: "app-platform")
        urlRequest.timeoutInterval = 300
        
        // Set Content-Type Header to multipart/form-data, this is equivalent to submitting form data with file upload in a web browser
        // And the boundary is also set here
        let boundary = "visikard+"+UUID().uuidString
        let data = createMultipartParamsBodyData(multipartDatas: multipartDatas, boundary: boundary)
        urlRequest.setValue("multipart/form-data; boundary=\(boundary)", forHTTPHeaderField: "Content-Type")
        urlRequest.setValue("\(data.count)", forHTTPHeaderField: "Content-Length")
        
        print("curl command: " + (urlRequest.cURL.count > 5000 ? "" : urlRequest.cURL))
        
        // Send a POST request to the URL, with the data we created earlier
        let uploadTask = session.uploadTask(with: urlRequest, from: data, completionHandler: { data, response, error in
            DispatchQueue.main.async {
                if let data = data {
                    do {
                        let dict = try JSONSerialization.jsonObject(with: data, options: [])
                        completion(nil, dict)
                    }catch let error {
                        APIRequestHandleError.shared().handleError(error)
                        completion(nil, nil)
                    }
                }else{
                    APIRequestHandleError.shared().handleError(error)
                    completion(nil, nil)
                }
            }
        })
        uploadTask.resume()
    }
    
    func requestFormData(urlString:String,method:HttpMethod, bodyDatas:[String: Any]? = nil,contentType: ContentType = .urlFormEncoded, uploadProgessBlock:@escaping UploadProgessBlock = {(_) in}, _ completion:@escaping ResponseBlock,_ errorHandlerBlock:ErrorBlock?){
        guard let url = URL(string: urlString) else {
            completion(nil, nil)
            return
        }
        
        var urlRequest = URLRequest(url: url,timeoutInterval: Double.infinity)
//        urlRequest.addValue(X_TENANT, forHTTPHeaderField: "X-TENANT")
//        urlRequest.addValue(UserDataManager.shared().deviceID, forHTTPHeaderField: "deviceid")
        urlRequest.addValue(contentType.rawValue, forHTTPHeaderField: "content-type")
        urlRequest.addValue("application/json", forHTTPHeaderField: "accept")
//        urlRequest.addValue(UserDataManager.shared().token, forHTTPHeaderField: "token")
//        urlRequest.addValue(SupportFunction.getAppVersionString(), forHTTPHeaderField: "appVersion")
//        urlRequest.addValue(APP_PLATFORM, forHTTPHeaderField: "app-platform")

        urlRequest.httpMethod = method.rawValue
        if let params = bodyDatas {
            let parameters = contentType == .urlFormEncoded ? params.queryString : params.toJsonString
            urlRequest.httpBody = parameters.data(using: .utf8)
        }
        
        print("curl command: " + urlRequest.cURL)
        let task = URLSession.shared.dataTask(with: urlRequest) { data, response, error in
            guard let data = data else {
                APIRequestHandleError.shared().handleError(error)
                completion(nil, nil)
                return
            }
            do {
                let dict = try JSONSerialization.jsonObject(with: data, options: [])
                completion(nil, dict)
            }catch let error {
                APIRequestHandleError.shared().handleError(error)
                completion(nil, nil)
            }
        }

        task.resume()
    }
    
    func createMultipartParamsBodyData(multipartDatas: [MultipartData],boundary : String) -> Data  {
        ///Init data
        var data = Data()
        
        ///Bounds
        let startBound = "--\(boundary)"+ctrl
        let centerBound = ctrl+"--\(boundary)"+ctrl
        let endBound = ctrl+"--\(boundary)--"+ctrl
        
        //Depositions
        let contentDisposition = "Content-Disposition: form-data; name=\"%@\""+ctrl+ctrl
        let fileContentDisposition = "Content-Disposition: form-data; name=\"%@\"; filename=\"%@\""+ctrl
        
        ///Content Type
        let contentType = "Content-Type: %@"+ctrl+ctrl
        
        //Add data
        for (index,multipartData) in multipartDatas.enumerated() {
            if index == 0 {
                data.append(startBound.utf8Data)
            } else {
                data.append(centerBound.utf8Data)
            }
            
            if let multipartData = multipartData as? MultipartDataString {
                data.append(String(format: contentDisposition, multipartData.key).utf8Data)
                data.append(multipartData.payload.utf8Data)
            } else if let multipartData = multipartData as? MultipartDataFile {
                data.append(String(format: fileContentDisposition,multipartData.key,multipartData.fileName).utf8Data)
                data.append(String(format: contentType,multipartData.mineType).utf8Data)
                data.append(multipartData.payload)
            } else {
                debugPrint("Request with invalid payload")
                continue
            }
        }
        data.append(endBound.utf8Data)
        return data
    }
}

//MARK: - URLSessionDelegate
extension APIRequest  : URLSessionDelegate{
    
    func urlSession(_ session: URLSession, didReceive challenge: URLAuthenticationChallenge, completionHandler: @escaping (URLSession.AuthChallengeDisposition, URLCredential?) -> Void) {
        
        let serverTrust = challenge.protectionSpace.serverTrust
        let certificate = SecTrustGetCertificateAtIndex(serverTrust!, 0)
        
        if !self.enableSSL {
            let credential:URLCredential = URLCredential(trust: serverTrust!)
            completionHandler(.useCredential, credential)
        } else {
            // Set SSL policies for domain name check
            let policies = NSMutableArray();
            policies.add(SecPolicyCreateSSL(true, (challenge.protectionSpace.host as CFString)))
            SecTrustSetPolicies(serverTrust!, policies);
            
            // Evaluate server certificate
            var result: SecTrustResultType = SecTrustResultType(rawValue: 0)!
            SecTrustEvaluate(serverTrust!, &result)
            let isServerTrusted:Bool = result == SecTrustResultType.unspecified || result ==  SecTrustResultType.proceed
            
            // Get local and remote cert data
            let remoteCertificateData:NSData = SecCertificateCopyData(certificate!)
            let localCertificate:NSData = NSData(contentsOfFile: API_CERT_PATH!)!
            
            if (isServerTrusted && remoteCertificateData.isEqual(to: localCertificate as Data)) {
                let credential:URLCredential = URLCredential(trust: serverTrust!)
                completionHandler(.useCredential, credential)
            } else {
                completionHandler(.cancelAuthenticationChallenge, nil)
            }
        }
    }
}

//MARK: Encode
extension APIRequest {
    private func query(_ parameters: [String: Any]) -> String {
        var components: [(String, String)] = []
        
        for key in parameters.keys.sorted(by: <) {
            let value = parameters[key]!
            components += queryComponents(fromKey: key, value: value)
        }
        return components.map { "\($0)=\($1)" }.joined(separator: "&")
    }
    
    public func queryComponents(fromKey key: String, value: Any) -> [(String, String)] {
        var components: [(String, String)] = []
        
        if let dictionary = value as? [String: Any] {
            for (nestedKey, value) in dictionary {
                components += queryComponents(fromKey: "\(key)[\(nestedKey)]", value: value)
            }
        } else if let array = value as? [Any] {
            for value in array {
                components += queryComponents(fromKey: arrayEncoding.encode(key: key), value: value)
            }
        } else if let value = value as? NSNumber {
            if value.isBool {
                components.append((escape(key), escape(boolEncoding.encode(value: value.boolValue))))
            } else {
                components.append((escape(key), escape("\(value)")))
            }
        } else if let bool = value as? Bool {
            components.append((escape(key), escape(boolEncoding.encode(value: bool))))
        } else {
            components.append((escape(key), escape("\(value)")))
        }
        
        return components
    }
    public func escape(_ string: String) -> String {
        let generalDelimitersToEncode = ":#[]@" // does not include "?" or "/" due to RFC 3986 - Section 3.4
        let subDelimitersToEncode = "!$&'()*+,;="
        
        var allowedCharacterSet = CharacterSet.urlQueryAllowed
        allowedCharacterSet.remove(charactersIn: "\(generalDelimitersToEncode)\(subDelimitersToEncode)")
        
        var escaped = ""
        
        //==========================================================================================================
        //
        //  Batching is required for escaping due to an internal bug in iOS 8.1 and 8.2. Encoding more than a few
        //  hundred Chinese characters causes various malloc error crashes. To avoid this issue until iOS 8 is no
        //  longer supported, batching MUST be used for encoding. This introduces roughly a 20% overhead. For more
        //  info, please refer to:
        //
        //      - https://github.com/Alamofire/Alamofire/issues/206
        //
        //==========================================================================================================
        
        if #available(iOS 8.3, *) {
            escaped = string.addingPercentEncoding(withAllowedCharacters: allowedCharacterSet) ?? string
        } else {
            let batchSize = 50
            var index = string.startIndex
            
            while index != string.endIndex {
                let startIndex = index
                let endIndex = string.index(index, offsetBy: batchSize, limitedBy: string.endIndex) ?? string.endIndex
                let range = startIndex..<endIndex
                
                let substring = string[range]
                
                escaped += substring.addingPercentEncoding(withAllowedCharacters: allowedCharacterSet) ?? String(substring)
                
                index = endIndex
            }
        }
        
        return escaped
    }
    
    public enum BoolEncoding {
        case numeric, literal
        
        func encode(value: Bool) -> String {
            switch self {
            case .numeric:
                return value ? "1" : "0"
            case .literal:
                return value ? "true" : "false"
            }
        }
    }
    
    public enum ArrayEncoding {
        case brackets, noBrackets
        
        func encode(key: String) -> String {
            switch self {
            case .brackets:
                return "\(key)[]"
            case .noBrackets:
                return key
            }
        }
    }
}

extension NSNumber {
    fileprivate var isBool: Bool { return CFBooleanGetTypeID() == CFGetTypeID(self) }
}

//MARK: Upload Proxy Delegate
class URLSessionDataProxyDelegate : NSObject, URLSessionDataDelegate,URLSessionDelegate {
    
    var uploadProgessBlock : UploadProgessBlock?
    
    init(uploadProgessBlock : UploadProgessBlock?) {
        super.init()
        self.uploadProgessBlock = uploadProgessBlock
    }
    
    func urlSession(_ session: URLSession, task: URLSessionTask, didCompleteWithError error: Error?) {
        print("Complete upload with error " + error.debugDescription)
    }
    
    func urlSession(_ session: URLSession, dataTask: URLSessionDataTask, didReceive response: URLResponse, completionHandler: @escaping (URLSession.ResponseDisposition) -> Void) {
        print("didReceive response")
        completionHandler(.allow)
    }
    
    func urlSession(_ session: URLSession, task: URLSessionTask, didSendBodyData bytesSent: Int64, totalBytesSent: Int64, totalBytesExpectedToSend: Int64) {
        print("totalBytesExpectedToSend:\(totalBytesExpectedToSend) - totalBytesSent:\(totalBytesSent) - bytesSent:\(bytesSent)")
        guard let uploadProgessBlock = uploadProgessBlock else {return}
        
        uploadProgessBlock( totalBytesSent == 0 ? 0 : Double(bytesSent/totalBytesSent))
    }
    
    func urlSession(_ session: URLSession, didReceive challenge: URLAuthenticationChallenge, completionHandler: @escaping (URLSession.AuthChallengeDisposition, URLCredential?) -> Void) {
        
        let serverTrust = challenge.protectionSpace.serverTrust
        let certificate = SecTrustGetCertificateAtIndex(serverTrust!, 0)
        
        // Not check SSL
        let credential:URLCredential = URLCredential(trust: serverTrust!)
        completionHandler(.useCredential, credential)
        
        // Open this if want to check SSL
        /*
        // Set SSL policies for domain name check
        let policies = NSMutableArray();
        policies.add(SecPolicyCreateSSL(true, (challenge.protectionSpace.host as CFString)))
        SecTrustSetPolicies(serverTrust!, policies);
        
        // Evaluate server certificate
        var result: SecTrustResultType = SecTrustResultType(rawValue: 0)!
        SecTrustEvaluate(serverTrust!, &result)
        let isServerTrusted:Bool = result == SecTrustResultType.unspecified || result ==  SecTrustResultType.proceed
        
        // Get local and remote cert data
        let remoteCertificateData:NSData = SecCertificateCopyData(certificate!)
        let localCertificate:NSData = NSData(contentsOfFile: API_CERT_PATH!)!
        
        if (isServerTrusted && remoteCertificateData.isEqual(to: localCertificate as Data)) {
            let credential:URLCredential = URLCredential(trust: serverTrust!)
            completionHandler(.useCredential, credential)
        } else {
            completionHandler(.cancelAuthenticationChallenge, nil)
        }
         */
    }
    
}


//MARK: Multipart data model
class MultipartData {
    var key : String = ""
    fileprivate init(key : String) {
        self.key = key
    }
}

class MultipartDataFile : MultipartData {
    var fileName : String
    var payload : Data
    var mineType : String
    init(key: String, fileName: String, payload: Data, mineType: String){
        self.fileName = fileName
        self.payload = payload
        self.mineType = mineType
        super.init(key: key)
    }
}

class MultipartDataString : MultipartData{
    var payload : String
    
    init(key: String,payload: String){
        self.payload = payload
        super.init(key: key)
    }
}
